/*
 * DISCLAIMER AND LIMITATION OF LIABILITY: Opto Engineering does not make or
 * give any representation or warranty with respect to the usefulness or the
 * efficiency of this software, it being understood that the degree of success
 * with which equipment, software, modifications, and other materials can be
 * applied to data processing is dependent upon many factors, many of which
 * are not under Opto Engineering's control.  ACCORDINGLY, THIS SOFTWARE IS
 * PROVIDED 'AS IS' WITHOUT EXPRESS OR IMPLIED WARRANTIES, INCLUDING NO
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  THIS SOFTWARE IS PROVIDED GRATUITOUSLY AND, ACCORDINGLY,
 * OPTO ENGINEERING SHALL NOT BE LIABLE UNDER ANY THEORY FOR ANY DAMAGES
 * SUFFERED BY YOU OR ANY USER OF THE SOFTWARE.  OPTO ENGINEERING WILL NOT
 * SUPPORT THIS SOFTWARE AND IS UNDER NO OBLIGATION TO ISSUE UPDATES TO THIS
 * SOFTWARE.
 *
 * WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, NEITHER OPTO ENGINEERING
 * NOR ITS SUPPLIERS SHALL BE LIABLE FOR (a) INCIDENTAL, CONSEQUENTIAL,
 * SPECIAL OR INDIRECT DAMAGES OF ANY SORT, WHETHER ARISING IN TORT, CONTRACT
 * OR OTHERWISE, EVEN IF OPTO ENGINEERING HAS BEEN INFORMED OF THE POSSIBILITY
 * OF SUCH DAMAGES, OR (b) FOR ANY CLAIM BY ANY OTHER PARTY.  SOME STATES DO
 * NOT ALLOW THE EXCLUSION OR LIMITATION OF INCIDENTAL OR CONSEQUENTIAL
 * DAMAGES, SO THIS LIMITATION AND EXCLUSION MAY NOT APPLY TO YOU.
 *
 * Written with Microsoft Visual C++ 2013 Express.
 *
 * Date: 25 APR 2018 - Version: 1.0
 * Author: Sergio Sigala
 */

#include "stdafx.h"

#include <ws2tcpip.h>

#include "modbus.h"

/* table of CRC values for loworder byte */
static unsigned char modbus_crc_table_lo[256] = {
	0x00, 0xC0, 0xC1, 0x01, 0xC3, 0x03, 0x02, 0xC2, 0xC6, 0x06, 0x07, 0xC7, 0x05, 0xC5, 0xC4, 0x04,
	0xCC, 0x0C, 0x0D, 0xCD, 0x0F, 0xCF, 0xCE, 0x0E, 0x0A, 0xCA, 0xCB, 0x0B, 0xC9, 0x09, 0x08, 0xC8,
	0xD8, 0x18, 0x19, 0xD9, 0x1B, 0xDB, 0xDA, 0x1A, 0x1E, 0xDE, 0xDF, 0x1F, 0xDD, 0x1D, 0x1C, 0xDC,
	0x14, 0xD4, 0xD5, 0x15, 0xD7, 0x17, 0x16, 0xD6, 0xD2, 0x12, 0x13, 0xD3, 0x11, 0xD1, 0xD0, 0x10,
	0xF0, 0x30, 0x31, 0xF1, 0x33, 0xF3, 0xF2, 0x32, 0x36, 0xF6, 0xF7, 0x37, 0xF5, 0x35, 0x34, 0xF4,
	0x3C, 0xFC, 0xFD, 0x3D, 0xFF, 0x3F, 0x3E, 0xFE, 0xFA, 0x3A, 0x3B, 0xFB, 0x39, 0xF9, 0xF8, 0x38,
	0x28, 0xE8, 0xE9, 0x29, 0xEB, 0x2B, 0x2A, 0xEA, 0xEE, 0x2E, 0x2F, 0xEF, 0x2D, 0xED, 0xEC, 0x2C,
	0xE4, 0x24, 0x25, 0xE5, 0x27, 0xE7, 0xE6, 0x26, 0x22, 0xE2, 0xE3, 0x23, 0xE1, 0x21, 0x20, 0xE0,
	0xA0, 0x60, 0x61, 0xA1, 0x63, 0xA3, 0xA2, 0x62, 0x66, 0xA6, 0xA7, 0x67, 0xA5, 0x65, 0x64, 0xA4,
	0x6C, 0xAC, 0xAD, 0x6D, 0xAF, 0x6F, 0x6E, 0xAE, 0xAA, 0x6A, 0x6B, 0xAB, 0x69, 0xA9, 0xA8, 0x68,
	0x78, 0xB8, 0xB9, 0x79, 0xBB, 0x7B, 0x7A, 0xBA, 0xBE, 0x7E, 0x7F, 0xBF, 0x7D, 0xBD, 0xBC, 0x7C,
	0xB4, 0x74, 0x75, 0xB5, 0x77, 0xB7, 0xB6, 0x76, 0x72, 0xB2, 0xB3, 0x73, 0xB1, 0x71, 0x70, 0xB0,
	0x50, 0x90, 0x91, 0x51, 0x93, 0x53, 0x52, 0x92, 0x96, 0x56, 0x57, 0x97, 0x55, 0x95, 0x94, 0x54,
	0x9C, 0x5C, 0x5D, 0x9D, 0x5F, 0x9F, 0x9E, 0x5E, 0x5A, 0x9A, 0x9B, 0x5B, 0x99, 0x59, 0x58, 0x98,
	0x88, 0x48, 0x49, 0x89, 0x4B, 0x8B, 0x8A, 0x4A, 0x4E, 0x8E, 0x8F, 0x4F, 0x8D, 0x4D, 0x4C, 0x8C,
	0x44, 0x84, 0x85, 0x45, 0x87, 0x47, 0x46, 0x86, 0x82, 0x42, 0x43, 0x83, 0x41, 0x81, 0x80, 0x40
};

/* table of CRC values for highorder byte */
static unsigned char modbus_crc_table_hi[256] = {
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40
};

/* calculate the CRC16 */
int modbus_calc_crc16(unsigned char *p_buf, unsigned short num_byte, unsigned char *p_crc_lo, unsigned char *p_crc_hi)
{
	int err = 0;

	if (p_buf == NULL) { err = 1; printf("bad parameter p_buf\n"); goto exit; }
	if (num_byte <= 0) { err = 1; printf("bad parameter num_byte\n"); goto exit; }
	if (p_crc_lo == NULL) { err = 1; printf("bad parameter p_crc_lo\n"); goto exit; }
	if (p_crc_hi == NULL) { err = 1; printf("bad parameter p_crc_hi\n"); goto exit; }

	unsigned char crc_lo = 0xFF; /* low byte of CRC initialized */
	unsigned char crc_hi = 0xFF; /* high byte of CRC initialized */
	unsigned char index; /* will index into CRC lookup table */

	while (num_byte-- > 0) { /* pass through message buffer */
		index = (crc_lo ^ *p_buf++); /* update the CRC */
		crc_lo = (crc_hi ^ modbus_crc_table_hi[index]);
		crc_hi = modbus_crc_table_lo[index];
	}

	/* emit CRC */
	*p_crc_lo = crc_lo;
	*p_crc_hi = crc_hi;

exit:
	return err;
}

/* send MODBUS request and wait for reply or timeout */
int modbus_transfer(modbus_t *p_modbus, int dev_cmd, int skip_tx, unsigned char *p_tx, int num_tx, unsigned char *p_rx, int num_rx)
{
	int err = 0;
	unsigned char tx_msg[256];
	unsigned char rx_msg[256];
	unsigned char tx_crc_lo;
	unsigned char tx_crc_hi;
	unsigned char rx_crc_lo;
	unsigned char rx_crc_hi;
	int j;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_modbus->dev_adr < 0) { err = 1; printf("bad parameter dev_adr\n"); goto exit; }
	if (dev_cmd < 0) { err = 1; printf("bad parameter dev_cmd\n"); goto exit; }
	if (!skip_tx && p_tx == NULL) { err = 1; printf("bad parameter p_tx\n"); goto exit; }
	if (num_tx < 0) { err = 1; printf("bad parameter num_tx\n"); goto exit; }
	if (p_rx == NULL) { err = 1; printf("bad parameter p_rx\n"); goto exit; }
	if (num_rx < 0) { err = 1; printf("bad parameter num_rx\n"); goto exit; }

	if (p_modbus->ser_initialized != 0 && p_modbus->mode == MODBUS_MODE_SER_485) {
		if (!skip_tx) {
			tx_msg[0] = p_modbus->dev_adr;	/* ADR */
			tx_msg[1] = dev_cmd;			/* CMD */

			unsigned char *p_tx_msg = &tx_msg[2];
			unsigned char *p_tx_ptr = p_tx;
			for (j = 0; j < num_tx; j++) {
				*p_tx_msg++ = *p_tx_ptr++; /* VALUE */
			}

			if ((err = modbus_calc_crc16(tx_msg, (num_tx + 2), &tx_crc_lo, &tx_crc_hi)) != 0) {
				printf("error in modbus_calc_crc16() call\n");
				goto exit;
			}

			tx_msg[num_tx + 2] = tx_crc_lo; /* CRC_LO */
			tx_msg[num_tx + 3] = tx_crc_hi; /* CRC_HI */

			DWORD tx_length = -1;

			/* send data */
			if (WriteFile(p_modbus->ser_handle, tx_msg, (num_tx + 4), &tx_length, NULL) == 0) {
				err = 1;
				printf("error %ld in WriteFile() call\n", GetLastError());
				goto exit;
			}

			/* dump sent data */
			if (0) {
				printf("sent data: ");
				for (j = 0; j < (int)tx_length; j++) {
					printf("%02X", tx_msg[j]);
				}
				printf(" (%d bytes) ", tx_length);
			}

			if (tx_length != (num_tx + 4)) {
				err = 1;
				printf("transmit error: %d of %d bytes\n", tx_length, (num_tx + 4));
				goto exit;
			}
		}

		/* 10ms delay */
		Sleep(10);

		DWORD rx_length = -1;

		/* receive data */
		if (ReadFile(p_modbus->ser_handle, rx_msg, (num_rx + 4), &rx_length, NULL) == 0) {
			err = 1;
			printf("error %ld in ReadFile() call\n", GetLastError());
			goto exit;
		}

		/* dump received data */
		if (0) {
			printf("received data: ");
			for (j = 0; j < (int)rx_length; j++) {
				printf("%02X", rx_msg[j]);
			}
			printf(" (%d bytes)\n", rx_length);
		}

		if (rx_length != (num_rx + 4)) {
			err = 1;
			printf("receive error: %d of %d bytes\n", rx_length, (num_rx + 4));
			goto exit;
		}

		if ((err = modbus_calc_crc16(rx_msg, (num_rx + 2), &rx_crc_lo, &rx_crc_hi)) != 0) {
			printf("error in modbus_calc_crc16() call\n");
			goto exit;
		}

		if (rx_msg[num_rx + 2] != rx_crc_lo || rx_msg[num_rx + 3] != rx_crc_hi) {
			err = 1;
			printf("bad CRC in received data\n");
			goto exit;
		}

		if (rx_msg[0] != p_modbus->dev_adr) { /* ADR */
			err = 1;
			printf("bad address in received data\n");
			goto exit;
		}

		if (rx_msg[1] != dev_cmd) { /* CMD */
			err = 1;
			printf("bad command in received data\n");
			goto exit;
		}

		unsigned char *p_rx_msg = &rx_msg[2];
		unsigned char *p_rx_ptr = p_rx;
		for (j = 0; j < num_rx; j++) {
			*p_rx_ptr++ = *p_rx_msg++; /* VALUE */
		}
	}
	else if (p_modbus->eth_initialized != 0 && p_modbus->mode == MODBUS_MODE_ETH_TCP) {
		int message_length_tx = (num_tx + 2);

		tx_msg[0] = 0;					/* transaction identifier MSB */
		tx_msg[1] = 0;					/* transaction identifier LSB */
		tx_msg[2] = 0;					/* protocolo identifier MSB */
		tx_msg[3] = 0;					/* protocolo identifier LSB */
		tx_msg[4] = 0;					/* message length MSB */
		tx_msg[5] = message_length_tx;	/* message length LSB */
		tx_msg[6] = p_modbus->dev_adr;	/* ADR */
		tx_msg[7] = dev_cmd;			/* CMD */

		unsigned char *p_tx_msg = &tx_msg[8];
		unsigned char *p_tx_ptr = p_tx;
		for (j = 0; j < num_tx; j++) {
			*p_tx_msg++ = *p_tx_ptr++; /* VALUE */
		}

		int tx_length;

		/* send data */
		if ((tx_length = send(p_modbus->eth_socket, (const char *)tx_msg, (num_tx + 8), 0)) == SOCKET_ERROR) {
			err = 1;
			printf("error %ld in send() call\n", WSAGetLastError());
			goto exit;
		}

		if (tx_length != (num_tx + 8)) {
			err = 1;
			printf("transmit error: %d of %d bytes\n", tx_length, (num_tx + 8));
			goto exit;
		}

		/* 10ms delay */
		Sleep(10);

		int rx_length;

		/* peeks at the incoming data */
		if ((rx_length = recv(p_modbus->eth_socket, (char *)&rx_msg, (num_rx + 8), MSG_PEEK)) == SOCKET_ERROR) {
			err = 1;
			printf("error %ld in recv() call\n", WSAGetLastError());
			goto exit;
		}

		/* dump sent data */
		if (0) {
			printf("sent data: ");
			for (j = 0; j < tx_length; j++) {
				printf("%02X", tx_msg[j]);
			}
			printf(" (%d bytes) ", tx_length);
		}

		/* dump received data */
		if (0) {
			printf("received data: ");
			for (j = 0; j < rx_length; j++) {
				printf("%02X", rx_msg[j]);
			}
			printf(" (%d bytes)\n", rx_length);
		}

		if (rx_length != (num_rx + 8)) {
			err = 1;
			printf("receive error: %d of %d bytes\n", rx_length, (num_rx + 8));
			goto exit;
		}

		/* receive data */
		if ((rx_length = recv(p_modbus->eth_socket, (char *)&rx_msg, (num_rx + 8), 0)) == SOCKET_ERROR) {
			err = 1;
			printf("error %ld in recv() call\n", WSAGetLastError());
			goto exit;
		}

		if (rx_length != (num_rx + 8)) {
			err = 1;
			printf("receive error: %d of %d bytes\n", rx_length, (num_rx + 8));
			goto exit;
		}

		if (rx_msg[0] != 0 || rx_msg[1] != 0) { /* transaction identifier */
			err = 1;
			printf("bad transaction identifier in received data 0x%02X 0x%02X\n", rx_msg[0], rx_msg[1]);
			goto exit;
		}

		if (rx_msg[2] != 0 || rx_msg[3] != 0) { /* protocol identifier */
			err = 1;
			printf("bad protocol identifier in received data 0x%02X 0x%02X\n", rx_msg[2], rx_msg[3]);
			goto exit;
		}

		int message_length_rx = ((rx_msg[4] << 16) | rx_msg[5]);

		if ((message_length_rx + 6) != rx_length) { /* message length */
			err = 1;
			printf("bad message length in received data 0x%02X 0x%02X\n", rx_msg[4], rx_msg[5]);
			goto exit;
		}

		if (rx_msg[6] != p_modbus->dev_adr) { /* ADR */
			err = 1;
			printf("bad address in received data\n");
			goto exit;
		}

		if (rx_msg[7] != dev_cmd) { /* CMD */
			err = 1;
			printf("bad command in received data\n");
			goto exit;
		}

		unsigned char *p_rx_msg = &rx_msg[8];
		unsigned char *p_rx_ptr = p_rx;
		for (j = 0; j < num_rx; j++) {
			*p_rx_ptr++ = *p_rx_msg++; /* VALUE */
		}
	}
	else {
		/* unsupported media or protocol */
	}

	/* 10ms delay */
	Sleep(10);

exit:
	return err;
}

/* read a group of MODBUS registers */
int modbus_reg_rd_group(modbus_t *p_modbus, int reg_adr, int num_reg, int *p_reg_val, int has_sign)
{
	int err = 0;
	unsigned char tx_data[256];
	unsigned char rx_data[256];
	int j;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_modbus->dev_adr < 0) { err = 1; printf("bad parameter dev_adr\n"); goto exit; }
	if (reg_adr < 0) { err = 1; printf("bad parameter reg_adr\n"); goto exit; }
	if (num_reg <= 0) { err = 1; printf("bad parameter num_reg\n"); goto exit; }
	if (p_reg_val == NULL) { err = 1; printf("bad parameter p_reg_val\n"); goto exit; }

	tx_data[0] = ((reg_adr >> 8) & 0xff); /* starting address MSB */
	tx_data[1] = ((reg_adr >> 0) & 0xff); /* starting address LSB */
	tx_data[2] = ((num_reg >> 8) & 0xff); /* number of registers MSB */
	tx_data[3] = ((num_reg >> 0) & 0xff); /* number of registers LSB */

	int dev_cmd = 0x03; /* Read Holding Registers */
	int num_tx = 4;
	int num_rx = (1 + num_reg * 2);

	if ((err = modbus_transfer(p_modbus, dev_cmd, 0, tx_data, num_tx, rx_data, num_rx)) != 0) {
		printf("error in modbus_transfer() call\n");
		goto exit;
	}

	if (rx_data[0] != (num_reg * 2)) {
		err = 1;
		printf("bad received byte count\n");
		goto exit;
	}

	/* unpack data */
	unsigned char *p_byte = &rx_data[1];
	for (j = 0; j < num_reg; j++) {
		if (has_sign) p_reg_val[j] = ((((p_byte[0] << 8) | p_byte[1]) << 16) >> 16); /* do sign extension */
		else p_reg_val[j] = ((p_byte[0] << 8) | p_byte[1]); /* no sign extension */
		p_byte += 2;
	}

exit:
	return err;
}

/* write a group of MODBUS registers */
int modbus_reg_wr_group(modbus_t *p_modbus, int reg_adr, int num_reg, int *p_reg_val)
{
	int err = 0;
	unsigned char tx_data[256];
	unsigned char rx_data[256];
	int j;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_modbus->dev_adr < 0) { err = 1; printf("bad parameter dev_adr\n"); goto exit; }
	if (reg_adr < 0) { err = 1; printf("bad parameter reg_adr\n"); goto exit; }
	if (num_reg <= 0) { err = 1; printf("bad parameter num_reg\n"); goto exit; }
	if (p_reg_val == NULL) { err = 1; printf("bad parameter p_reg_val\n"); goto exit; }

	tx_data[0] = ((reg_adr >> 8) & 0xff); /* starting address MSB */
	tx_data[1] = ((reg_adr >> 0) & 0xff); /* starting address LSB */
	tx_data[2] = ((num_reg >> 8) & 0xff); /* number of registers MSB */
	tx_data[3] = ((num_reg >> 0) & 0xff); /* number of registers LSB */
	tx_data[4] = (num_reg * 2); /* byte count */

	/* pack data */
	unsigned char *p_byte = &tx_data[5];
	for (j = 0; j < num_reg; j++) {
		p_byte[0] = ((p_reg_val[j] >> 8) & 0xff); /* data MSB */
		p_byte[1] = ((p_reg_val[j] >> 0) & 0xff); /* data LSB */
		p_byte += 2;
	}

	int dev_cmd = 0x10; /* Write Multiple Registers */
	int num_tx = (5 + num_reg * 2);
	int num_rx = 4;

	if ((err = modbus_transfer(p_modbus, dev_cmd, 0, tx_data, num_tx, rx_data, num_rx)) != 0) {
		printf("error in modbus_transfer() call\n");
		goto exit;
	}

	if (rx_data[0] != tx_data[0]) {
		err = 1;
		printf("bad received starting address MSB\n");
		goto exit;
	}

	if (rx_data[1] != tx_data[1]) {
		err = 1;
		printf("bad received starting address LSB\n");
		goto exit;
	}

	if (rx_data[2] != tx_data[2]) {
		err = 1;
		printf("bad received number of registers MSB\n");
		goto exit;
	}

	if (rx_data[3] != tx_data[3]) {
		err = 1;
		printf("bad received number of registers LSB\n");
		goto exit;
	}

exit:
	return err;
}

/* write a single MODBUS register */
int modbus_reg_wr_single(modbus_t *p_modbus, int reg_adr, int reg_val)
{
	int err = 0;
	unsigned char tx_data[256];
	unsigned char rx_data[256];

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_modbus->dev_adr < 0) { err = 1; printf("bad parameter dev_adr\n"); goto exit; }
	if (reg_adr < 0) { err = 1; printf("bad parameter reg_adr\n"); goto exit; }

	tx_data[0] = ((reg_adr >> 8) & 0xff); /* register address MSB */
	tx_data[1] = ((reg_adr >> 0) & 0xff); /* register address LSB */
	tx_data[2] = ((reg_val >> 8) & 0xff); /* register value MSB */
	tx_data[3] = ((reg_val >> 0) & 0xff); /* register value LSB */

	int dev_cmd = 0x06; /* Write Single Register */
	int num_tx = 4;
	int num_rx = 4;

	if ((err = modbus_transfer(p_modbus, dev_cmd, 0, tx_data, num_tx, rx_data, num_rx)) != 0) {
		printf("error in modbus_transfer() call\n");
		goto exit;
	}

	if (rx_data[0] != tx_data[0]) {
		err = 1;
		printf("bad received register address MSB\n");
		goto exit;
	}

	if (rx_data[1] != tx_data[1]) {
		err = 1;
		printf("bad received register address LSB\n");
		goto exit;
	}

	if (rx_data[2] != tx_data[2]) {
		err = 1;
		printf("bad received register value MSB\n");
		goto exit;
	}

	if (rx_data[3] != tx_data[3]) {
		err = 1;
		printf("bad received register value LSB\n");
		goto exit;
	}

exit:
	return err;
}

/* initialize data structures */
int modbus_init(modbus_t *p_modbus)
{
	int err = 0;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	p_modbus->ser_initialized = 0;
	p_modbus->eth_initialized = 0;

exit:
	return err;
}

/* open serial port and set communication parameters */
int modbus_open_ser(modbus_t *p_modbus, LPCWSTR p_port)
{
	int err = 0;
	DCB dcb;
	COMMTIMEOUTS cmt;
	int speed = 9600;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_port == NULL) { err = 1; printf("bad parameter p_port\n"); goto exit; }
	if (speed != 9600 && speed != 19200) { err = 1; printf("bad parameter speed\n"); goto exit; }

	wprintf(L"opening serial connection (port %s at %d bps)\n", p_port, speed);

	if ((p_modbus->ser_handle = CreateFile(p_port, GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING, 0, 0)) == INVALID_HANDLE_VALUE) {
		err = 1;
		printf("error %ld in CreateFile() call\n", GetLastError());
		goto exit;
	}

	/* create the DCB first, set every field of the DCB to 0 */
	FillMemory(&dcb, sizeof(dcb), 0);

	/* set the length of the DCB */
	dcb.DCBlength = sizeof(dcb);

	/* build the DCB */
	if (BuildCommDCB(((speed == 9600) ? L"9600,E,8,1" : L"19200,E,8,1"), &dcb) == 0) {
		err = 1;
		printf("error %ld in BuildCommDCB() call\n", GetLastError());
		goto exit;
	}

	/* set the state of the handle to be dcb */
	if (SetCommState(p_modbus->ser_handle, &dcb) == 0) {
		err = 1;
		printf("error %ld in SetCommState() call\n", GetLastError());
		goto exit;
	}

	/* set the buffer size */
	if (SetupComm(p_modbus->ser_handle, 1024 /* in queue */, 1024 /* out queue */) == 0) {
		err = 1;
		printf("error %ld in SetupComm() call\n", GetLastError());
		goto exit;
	}

	/* the maximum amount of time allowed to pass between the arrival of two bytes on the read line (in ms) */
	cmt.ReadIntervalTimeout = 20; //50

	/* value used to calculate the total time needed for a read operation which is (num bytes to read) * (timeout) in ms */
	cmt.ReadTotalTimeoutMultiplier = 20;

	/* value added to the previous one to generate the timeout value for a single read operation (in ms) */
	cmt.ReadTotalTimeoutConstant = 20;

	/* value used to calculate the total time needed for a write operation which is (num bytes to write) * (timeout) in ms */
	cmt.WriteTotalTimeoutMultiplier = 10;

	/* value added to the previous one to generate the timeout value for a single write operation (in ms) */
	cmt.WriteTotalTimeoutConstant = 10;

	/* set the timeouts of handle */
	if (SetCommTimeouts(p_modbus->ser_handle, &cmt) == 0) {
		err = 1;
		printf("error %ld in SetCommTimeouts() call\n", GetLastError());
		goto exit;
	}

	/* initialization done */
	p_modbus->ser_initialized = 1;

exit:
	return err;
}

/* open ethernet port and set communication parameters */
int modbus_open_eth(modbus_t *p_modbus, char *p_board_ip_address, char *p_board_port_number)
{
	int err = 0;
	int result;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_board_ip_address == NULL) { err = 1; printf("bad parameter p_board_ip_address\n"); goto exit; }

	printf("opening ethernet connection (IP %s port %s)\n", p_board_ip_address, p_board_port_number);

	/* create a socket for connecting to server */
	if ((p_modbus->eth_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == INVALID_SOCKET) {
		err = 1;
		printf("error %ld in socket() call\n", WSAGetLastError());
		goto exit;
	}

	sockaddr_in clientService;
	clientService.sin_family = AF_INET;
	clientService.sin_port = htons(atoi(p_board_port_number));

	/* convert address from string to numeric */
	if ((result = inet_pton(AF_INET, p_board_ip_address, &clientService.sin_addr)) != 1) {
		err = 1;
		printf("error %d/%ld in inet_pton() call\n", result, WSAGetLastError());
		goto exit;
	}

	/* connect to server */
	if ((result = connect(p_modbus->eth_socket, (SOCKADDR *) &clientService, sizeof(clientService))) == SOCKET_ERROR) {
		err = 1;
		printf("error %d/%ld in connect() call\n", result, WSAGetLastError());
		goto exit;
	}

	/* initialization done */
	p_modbus->eth_initialized = 1;

exit:
	return err;
}

/* close serial port */
int modbus_close_ser(modbus_t *p_modbus)
{
	int err = 0;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	if (p_modbus->ser_initialized != 0) {
		printf("closing serial connection\n");

		/* close the handle, thus releasing the device */
		if (CloseHandle(p_modbus->ser_handle) == 0) {
			err = 1;
			printf("error %ld in CloseHandle() call\n", GetLastError());
			goto exit;
		}
	}

exit:
	return err;
}

/* close ethernet port */
int modbus_close_eth(modbus_t *p_modbus)
{
	int err = 0;
	int result;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	if (p_modbus->eth_initialized != 0) {
		printf("closing ethernet connection\n");

		/* close the socket when finished exchanging datagrams */
		if ((result = closesocket(p_modbus->eth_socket)) == SOCKET_ERROR) {
			err = 1;
			printf("error %d/%ld in closesocket() call\n", result, WSAGetLastError());
			goto exit;
		}
	}

exit:
	return err;
}

/* select communication port */
int modbus_sel_mode(modbus_t *p_modbus, modbus_mode_t mode)
{
	int err = 0;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	p_modbus->mode = mode;

exit:
	return err;
}

/* select device addres */
int modbus_sel_addr(modbus_t *p_modbus, int addr)
{
	int err = 0;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	p_modbus->dev_adr = addr;

exit:
	return err;
}

/* file ends here */
