/*
 * DISCLAIMER AND LIMITATION OF LIABILITY: Opto Engineering does not make or
 * give any representation or warranty with respect to the usefulness or the
 * efficiency of this software, it being understood that the degree of success
 * with which equipment, software, modifications, and other materials can be
 * applied to data processing is dependent upon many factors, many of which
 * are not under Opto Engineering's control.  ACCORDINGLY, THIS SOFTWARE IS
 * PROVIDED 'AS IS' WITHOUT EXPRESS OR IMPLIED WARRANTIES, INCLUDING NO
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  THIS SOFTWARE IS PROVIDED GRATUITOUSLY AND, ACCORDINGLY,
 * OPTO ENGINEERING SHALL NOT BE LIABLE UNDER ANY THEORY FOR ANY DAMAGES
 * SUFFERED BY YOU OR ANY USER OF THE SOFTWARE.  OPTO ENGINEERING WILL NOT
 * SUPPORT THIS SOFTWARE AND IS UNDER NO OBLIGATION TO ISSUE UPDATES TO THIS
 * SOFTWARE.
 *
 * WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, NEITHER OPTO ENGINEERING
 * NOR ITS SUPPLIERS SHALL BE LIABLE FOR (a) INCIDENTAL, CONSEQUENTIAL,
 * SPECIAL OR INDIRECT DAMAGES OF ANY SORT, WHETHER ARISING IN TORT, CONTRACT
 * OR OTHERWISE, EVEN IF OPTO ENGINEERING HAS BEEN INFORMED OF THE POSSIBILITY
 * OF SUCH DAMAGES, OR (b) FOR ANY CLAIM BY ANY OTHER PARTY.  SOME STATES DO
 * NOT ALLOW THE EXCLUSION OR LIMITATION OF INCIDENTAL OR CONSEQUENTIAL
 * DAMAGES, SO THIS LIMITATION AND EXCLUSION MAY NOT APPLY TO YOU.
 *
 * Written with Microsoft Visual C++ 2013 Express.
 *
 * Date: 25 APR 2018 - Version: 1.0
 * Author: Sergio Sigala
 */

#include "stdafx.h"

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>

#include "main_app.h"
#include "modbus.h"
#include "dev_ltdv6ch.h"
#include "menu_ltdv6ch.h"
#include "menu.h"

/* read all the general purpose registers */
int ltdv6ch_menu_read_gen_reg(modbus_t *p_modbus)
{
	int err = 0;
	ltdv6ch_reg_rd_t reg;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	if ((err = ltdv6ch_gen_reg_rd(p_modbus, &reg, 1 /* print */)) != 0) {
		printf("error in ltdv6ch_gen_reg_rd() call\n");
		goto exit;
	}

exit:
	return err;
}

/* write all the general purpose registers */
int ltdv6ch_menu_write_gen_reg(modbus_t *p_modbus, int mode)
{
	int err = 0;
	ltdv6ch_reg_wr_t reg;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	if (mode == 0) { /* factory configuration */
		reg.inp_flt0	= 4;
		reg.inp_flt1	= 4;
		reg.inp_flt2	= 4;
		reg.inp_flt3	= 4;
		reg.inp_sel0	= 0;
		reg.inp_sel1	= 0;
		reg.inp_sel2	= 0;
		reg.inp_sel3	= 0;
		reg.inp_sel4	= 0;
		reg.inp_sel5	= 0;
		reg.inp_sel6	= 0;
		reg.inp_sel7	= 0;
		reg.gen_delay0	= 0;
		reg.gen_pulse0	= 100;
		reg.gen_delay1	= 0;
		reg.gen_pulse1	= 100;
		reg.gen_delay2	= 0;
		reg.gen_pulse2	= 100;
		reg.gen_delay3	= 0;
		reg.gen_pulse3	= 100;
		reg.gen_delay4	= 0;
		reg.gen_pulse4	= 100;
		reg.gen_delay5	= 0;
		reg.gen_pulse5	= 100;
		reg.gen_delay6	= 0;
		reg.gen_pulse6	= 100;
		reg.gen_delay7	= 0;
		reg.gen_pulse7	= 100;
		reg.out_sel0	= 0;
		reg.out_sel1	= 0;
		reg.out_sel2	= 0;
		reg.out_sel3	= 0;
		reg.out_sel4	= 0;
		reg.out_sel5	= 0;
		reg.out_sel6	= 0;
		reg.out_sel7	= 0;
		reg.cur_set0	= 36;
		reg.cur_set1	= 36;
		reg.cur_set2	= 36;
		reg.cur_set3	= 36;
		reg.cur_set4	= 36;
		reg.cur_set5	= 36;
		reg.ton_max		= 10;
		reg.toff_min	= 10;
	}
	else { /* test configuration */
		reg.inp_flt0	= 4;
		reg.inp_flt1	= 4;
		reg.inp_flt2	= 4;
		reg.inp_flt3	= 4;
		reg.inp_sel0	= 16;
		reg.inp_sel1	= 16;
		reg.inp_sel2	= 16;
		reg.inp_sel3	= 16;
		reg.inp_sel4	= 16;
		reg.inp_sel5	= 16;
		reg.inp_sel6	= 16;
		reg.inp_sel7	= 16;
		reg.gen_delay0	= 0;
		reg.gen_pulse0	= 100;
		reg.gen_delay1	= 0;
		reg.gen_pulse1	= 100;
		reg.gen_delay2	= 0;
		reg.gen_pulse2	= 100;
		reg.gen_delay3	= 0;
		reg.gen_pulse3	= 100;
		reg.gen_delay4	= 0;
		reg.gen_pulse4	= 100;
		reg.gen_delay5	= 0;
		reg.gen_pulse5	= 100;
		reg.gen_delay6	= 0;
		reg.gen_pulse6	= 100;
		reg.gen_delay7	= 0;
		reg.gen_pulse7	= 100;
		reg.out_sel0	= 1;
		reg.out_sel1	= 2;
		reg.out_sel2	= 4;
		reg.out_sel3	= 8;
		reg.out_sel4	= 16;
		reg.out_sel5	= 32;
		reg.out_sel6	= 64;
		reg.out_sel7	= 128;
		reg.cur_set0	= 75;
		reg.cur_set1	= 75;
		reg.cur_set2	= 75;
		reg.cur_set3	= 75;
		reg.cur_set4	= 75;
		reg.cur_set5	= 75;
		reg.ton_max		= 10;
		reg.toff_min	= 10;
	}

	if ((err = ltdv6ch_gen_reg_wr(p_modbus, &reg, 1)) != 0) {
		printf("error in ltdv6ch_gen_reg_wr() call\n");
		goto exit;
	}

exit:
	return err;
}

/* write board command register */
int ltdv6ch_menu_write_brd_cmd(modbus_t *p_modbus, int reg_val)
{
	int err = 0;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }

	if ((err = ltdv6ch_brd_cmd_wr(p_modbus, reg_val)) != 0) {
		printf("error in ltdv6ch_brd_cmd_wr() call\n");
		goto exit;
	}

exit:
	return err;
}

/* read all the general purpose registers and store data to disk file */
int ltdv6ch_menu_read_gen_reg_file(modbus_t *p_modbus, char *p_file_name)
{
	int err = 0;
	int print = 0;
	ltdv6ch_reg_rd_t reg_rd;
	ltdv6ch_reg_file_t reg_file;
	FILE *p_file = NULL;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_file_name == NULL) { err = 1; printf("bad parameter p_file_name\n"); goto exit; }

	/* open disk file for writing */
	if (fopen_s(&p_file, p_file_name, "wb") != 0) {
		printf("unable to open file %s for writing\n", p_file_name);
		goto exit;
	}

	/* read registers from device */
	printf("reading registers from device...\n");
	if ((err = ltdv6ch_gen_reg_rd(p_modbus, &reg_rd, print)) != 0) {
		printf("error in ltdv6ch_gen_reg_rd() call\n");
		goto exit;
	}
	printf("registers read from device\n");

	/* prepare data for writing to disk */
	reg_file.magic0 = LTDV6CH_FILE_MAGIC0;
	reg_file.magic1 = LTDV6CH_FILE_MAGIC1;
	reg_file.inp_flt0 = reg_rd.inp_flt0;
	reg_file.inp_flt1 = reg_rd.inp_flt1;
	reg_file.inp_flt2 = reg_rd.inp_flt2;
	reg_file.inp_flt3 = reg_rd.inp_flt3;
	reg_file.inp_sel0 = reg_rd.inp_sel0;
	reg_file.inp_sel1 = reg_rd.inp_sel1;
	reg_file.inp_sel2 = reg_rd.inp_sel2;
	reg_file.inp_sel3 = reg_rd.inp_sel3;
	reg_file.inp_sel4 = reg_rd.inp_sel4;
	reg_file.inp_sel5 = reg_rd.inp_sel5;
	reg_file.inp_sel6 = reg_rd.inp_sel6;
	reg_file.inp_sel7 = reg_rd.inp_sel7;
	reg_file.gen_delay0 = reg_rd.gen_delay0;
	reg_file.gen_pulse0 = reg_rd.gen_pulse0;
	reg_file.gen_delay1 = reg_rd.gen_delay1;
	reg_file.gen_pulse1 = reg_rd.gen_pulse1;
	reg_file.gen_delay2 = reg_rd.gen_delay2;
	reg_file.gen_pulse2 = reg_rd.gen_pulse2;
	reg_file.gen_delay3 = reg_rd.gen_delay3;
	reg_file.gen_pulse3 = reg_rd.gen_pulse3;
	reg_file.gen_delay4 = reg_rd.gen_delay4;
	reg_file.gen_pulse4 = reg_rd.gen_pulse4;
	reg_file.gen_delay5 = reg_rd.gen_delay5;
	reg_file.gen_pulse5 = reg_rd.gen_pulse5;
	reg_file.gen_delay6 = reg_rd.gen_delay6;
	reg_file.gen_pulse6 = reg_rd.gen_pulse6;
	reg_file.gen_delay7 = reg_rd.gen_delay7;
	reg_file.gen_pulse7 = reg_rd.gen_pulse7;
	reg_file.out_sel0 = reg_rd.out_sel0;
	reg_file.out_sel1 = reg_rd.out_sel1;
	reg_file.out_sel2 = reg_rd.out_sel2;
	reg_file.out_sel3 = reg_rd.out_sel3;
	reg_file.out_sel4 = reg_rd.out_sel4;
	reg_file.out_sel5 = reg_rd.out_sel5;
	reg_file.out_sel6 = reg_rd.out_sel6;
	reg_file.out_sel7 = reg_rd.out_sel7;
	reg_file.cur_set0 = reg_rd.cur_set0;
	reg_file.cur_set1 = reg_rd.cur_set1;
	reg_file.cur_set2 = reg_rd.cur_set2;
	reg_file.cur_set3 = reg_rd.cur_set3;
	reg_file.cur_set4 = reg_rd.cur_set4;
	reg_file.cur_set5 = reg_rd.cur_set5;
	reg_file.ton_max = reg_rd.ton_max;
	reg_file.toff_min = reg_rd.toff_min;

	/* write data to file */
	if (fwrite(&reg_file, sizeof(reg_file), 1, p_file) != 1) {
		printf("unable to write to file %s\n", p_file_name);
		goto exit;
	}
	printf("data written to file \"%s\"\n", p_file_name);

exit:
	if (p_file != NULL) fclose(p_file);
	return err;
}

/* write all the general purpose registers with data read from disk file */
int ltdv6ch_menu_write_gen_reg_file(modbus_t *p_modbus, char *p_file_name)
{
	int err = 0;
	int print = 0;
	ltdv6ch_reg_wr_t reg_wr;
	ltdv6ch_reg_file_t reg_file;
	FILE *p_file = NULL;

	if (p_modbus == NULL) { err = 1; printf("bad parameter p_modbus\n"); goto exit; }
	if (p_file_name == NULL) { err = 1; printf("bad parameter p_file_name\n"); goto exit; }

	/* open disk file for reading */
	if (fopen_s(&p_file, p_file_name, "rb") != 0) {
		printf("unable to open file %s for reading\n", p_file_name);
		goto exit;
	}

	/* read data from file */
	if (fread(&reg_file, sizeof(reg_file), 1, p_file) != 1) {
		printf("unable to read from file %s\n", p_file_name);
		goto exit;
	}
	printf("data read from file \"%s\"\n", p_file_name);

	/* check header */
	if ((reg_file.magic0 != LTDV6CH_FILE_MAGIC0) || (reg_file.magic1 != LTDV6CH_FILE_MAGIC1)) {
		printf("bad header in file %s\n", p_file_name);
		goto exit;
	}

	/* prepare data for writing to device */
	reg_wr.inp_flt0 = reg_file.inp_flt0;
	reg_wr.inp_flt1 = reg_file.inp_flt1;
	reg_wr.inp_flt2 = reg_file.inp_flt2;
	reg_wr.inp_flt3 = reg_file.inp_flt3;
	reg_wr.inp_sel0 = reg_file.inp_sel0;
	reg_wr.inp_sel1 = reg_file.inp_sel1;
	reg_wr.inp_sel2 = reg_file.inp_sel2;
	reg_wr.inp_sel3 = reg_file.inp_sel3;
	reg_wr.inp_sel4 = reg_file.inp_sel4;
	reg_wr.inp_sel5 = reg_file.inp_sel5;
	reg_wr.inp_sel6 = reg_file.inp_sel6;
	reg_wr.inp_sel7 = reg_file.inp_sel7;
	reg_wr.gen_delay0 = reg_file.gen_delay0;
	reg_wr.gen_pulse0 = reg_file.gen_pulse0;
	reg_wr.gen_delay1 = reg_file.gen_delay1;
	reg_wr.gen_pulse1 = reg_file.gen_pulse1;
	reg_wr.gen_delay2 = reg_file.gen_delay2;
	reg_wr.gen_pulse2 = reg_file.gen_pulse2;
	reg_wr.gen_delay3 = reg_file.gen_delay3;
	reg_wr.gen_pulse3 = reg_file.gen_pulse3;
	reg_wr.gen_delay4 = reg_file.gen_delay4;
	reg_wr.gen_pulse4 = reg_file.gen_pulse4;
	reg_wr.gen_delay5 = reg_file.gen_delay5;
	reg_wr.gen_pulse5 = reg_file.gen_pulse5;
	reg_wr.gen_delay6 = reg_file.gen_delay6;
	reg_wr.gen_pulse6 = reg_file.gen_pulse6;
	reg_wr.gen_delay7 = reg_file.gen_delay7;
	reg_wr.gen_pulse7 = reg_file.gen_pulse7;
	reg_wr.out_sel0 = reg_file.out_sel0;
	reg_wr.out_sel1 = reg_file.out_sel1;
	reg_wr.out_sel2 = reg_file.out_sel2;
	reg_wr.out_sel3 = reg_file.out_sel3;
	reg_wr.out_sel4 = reg_file.out_sel4;
	reg_wr.out_sel5 = reg_file.out_sel5;
	reg_wr.out_sel6 = reg_file.out_sel6;
	reg_wr.out_sel7 = reg_file.out_sel7;
	reg_wr.cur_set0 = reg_file.cur_set0;
	reg_wr.cur_set1 = reg_file.cur_set1;
	reg_wr.cur_set2 = reg_file.cur_set2;
	reg_wr.cur_set3 = reg_file.cur_set3;
	reg_wr.cur_set4 = reg_file.cur_set4;
	reg_wr.cur_set5 = reg_file.cur_set5;
	reg_wr.ton_max = reg_file.ton_max;
	reg_wr.toff_min = reg_file.toff_min;

	/* write registers to device */
	printf("writing registers to device...\n");
	if ((err = ltdv6ch_gen_reg_wr(p_modbus, &reg_wr, print)) != 0) {
		printf("error in ltdv6ch_gen_reg_wr() call\n");
		goto exit;
	}
	printf("registers written to device\n");

	/* store registers in non-volatile memory */
	printf("storing registers to non-volatile memory...\n");
	if ((err = ltdv6ch_menu_write_brd_cmd(p_modbus, 2)) != 0) {
		printf("error in ltdv6ch_menu_write_brd_cmd() call\n");
		goto exit;
	}
	printf("registers stored to non-volatile memory\n");

exit:
	if (p_file != NULL) fclose(p_file);
	return err;
}

/* program menu */
int ltdv6ch_menu_run()
{
	modbus_t modbus;
	int err = 0;
	int quit = 0;
	char file_name[] = "LTDV6CH.dat";

	/* communication parameters */
	LPCWSTR p_ser_port = MODBUS_COM;
	int modbus_addr = MODBUS_ADR;
	char *p_eth_board_ip_address = NULL;
	char *p_eth_board_port_number = NULL;

	if (p_ser_port == NULL) { err = 1; printf("bad parameter p_ser_port\n"); goto exit; }
	if (modbus_addr < 0) { err = 1; printf("bad parameter modbus_addr\n"); goto exit; }

	if ((err = modbus_init(&modbus)) != 0) {
		printf("error in modbus_init() call\n");
		goto exit;
	}

	if ((err = modbus_open_ser(&modbus, p_ser_port)) != 0) {
		printf("error in modbus_open_ser() call\n");
		goto exit;
	}

	if ((err = modbus_sel_mode(&modbus, MODBUS_MODE_SER_485)) != 0) {
		printf("error in modbus_sel_mode() call\n");
		goto exit;
	}

	if ((err = modbus_sel_addr(&modbus, modbus_addr)) != 0) {
		printf("error in modbus_sel_addr() call\n");
		goto exit;
	}

	while (!quit) {
		printf(
			"\n"
			"LTDV6CH demonstration program\n"
			"=============================\n"
			"\n"
			"0) Quit\n"
			"1) Read registers\n"
			"2) Write registers with factory configuration\n"
			"3) Write registers with test configuration\n"
			"4) Load registers from non-volatile memory\n"
			"5) Store registers to non-volatile memory\n"
			"6) Reboot device\n"
			"7) Read registers and save data to disk file \"%s\"\n"
			"8) Write registers with data loaded from disk file \"%s\" and store to non-volatile memory\n"
			"\n"
			"Choose: ", file_name, file_name
			);

		switch (menu_wait_selection()) {
		case 0:
			quit = 1;
			break;
		case 1:
			if ((err = ltdv6ch_menu_read_gen_reg(&modbus)) != 0) {
				printf("error in ltdv6ch_menu_read_gen_reg() call\n");
				goto exit;
			}
			break;
		case 2:
			if ((err = ltdv6ch_menu_write_gen_reg(&modbus, 0)) != 0) {
				printf("error in ltdv6ch_menu_write_gen_reg() call\n");
				goto exit;
			}
			break;
		case 3:
			if ((err = ltdv6ch_menu_write_gen_reg(&modbus, 1)) != 0) {
				printf("error in ltdv6ch_menu_write_gen_reg() call\n");
				goto exit;
			}
			break;
		case 4:
			if ((err = ltdv6ch_menu_write_brd_cmd(&modbus, 1)) != 0) {
				printf("error in ltdv6ch_menu_write_brd_cmd() call\n");
				goto exit;
			}
			break;
		case 5:
			if ((err = ltdv6ch_menu_write_brd_cmd(&modbus, 2)) != 0) {
				printf("error in ltdv6ch_menu_write_brd_cmd() call\n");
				goto exit;
			}
			break;
		case 6:
			if ((err = ltdv6ch_menu_write_brd_cmd(&modbus, 3)) != 0) {
				printf("error in ltdv6ch_menu_brd_cmd() call\n");
				goto exit;
			}
			break;
		case 7:
			if ((err = ltdv6ch_menu_read_gen_reg_file(&modbus, file_name)) != 0) {
				printf("error in ltdv6ch_menu_read_gen_reg_file() call\n");
				goto exit;
			}
			break;
		case 8:
			if ((err = ltdv6ch_menu_write_gen_reg_file(&modbus, file_name)) != 0) {
				printf("error in ltdv6ch_menu_write_gen_reg_file() call\n");
				goto exit;
			}
			break;
		default:
			break;
		}
	}

	if ((err = modbus_close_ser(&modbus)) != 0) {
		printf("error in modbus_close_ser() call\n");
		goto exit;
	}

	if ((err = modbus_close_eth(&modbus)) != 0) {
		printf("error in modbus_close_eth() call\n");
		goto exit;
	}

exit:
	return err;
}

/* file ends here */
